import { Request, Response } from 'express';
import { invoiceService } from './service';

function hasFeature(vendor: any, featureKey: string): boolean {
  return vendor?.features?.includes(featureKey);
}

export const invoicesController = {
  // POST /invoices
  async create(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'invoicing_enabled')) {
      return res.status(403).json({ error: 'Invoicing not available on your plan.' });
    }
    const { data, error } = await invoiceService.createInvoice(vendor.id, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.status(201).json({ success: true, data });
  },

  // GET /invoices/view/:public_code (public)
  async viewPublic(req: Request, res: Response) {
    const { public_code } = req.params;
    const { data, error } = await invoiceService.viewInvoice(public_code);
    if (error) return res.status(404).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // POST /invoices/:id/mark-paid (webhook)
  async markPaid(req: Request, res: Response) {
    const { id } = req.params;
    const { payment_gateway, amount, currency } = req.body;
    const { data, error } = await invoiceService.markPaid(id, payment_gateway, amount, currency);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // GET /invoices (vendor)
  async list(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'invoicing_enabled')) {
      return res.status(403).json({ error: 'Invoicing not available on your plan.' });
    }
    const { status, currency, date_range } = req.query;
    const filters: any = { status, currency, date_range };
    const { data, error } = await invoiceService.listInvoices(vendor.id, filters);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // GET /invoices/:id/download
  async downloadPdf(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'invoicing_enabled')) {
      return res.status(403).json({ error: 'Invoicing not available on your plan.' });
    }
    const { id } = req.params;
    const { data, error } = await invoiceService.getPdfUrl(id, vendor.id);
    if (error) return res.status(404).json({ error: error.message });
    return res.json({ success: true, pdf_url: data });
  },

  // DELETE /invoices/:id
  async delete(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'invoicing_enabled')) {
      return res.status(403).json({ error: 'Invoicing not available on your plan.' });
    }
    const { id } = req.params;
    const { success, error } = await invoiceService.deleteInvoice(id, vendor.id);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success });
  },

  // GET /transactions
  async listTransactions(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'invoicing_enabled')) {
      return res.status(403).json({ error: 'Invoicing not available on your plan.' });
    }
    const { data, error } = await invoiceService.listTransactions(vendor.id);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  }
}; 